// vtk_glad_sample.cpp : This file contains the 'main' function. Program execution begins and ends there.
//

#include <vtkSmartPointer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderer.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkSphereSource.h>
#include <vtkPolyDataMapper.h>
#include <vtkActor.h>
#include <vtkProperty.h>
#include <vtkOpenGLRenderWindow.h>

#include <iostream>
#include <thread>
#include <chrono>
#include <vtkglad/include/glad/gl.h>

int main(int argc, char* argv[])
{
    std::cout << "VTK GLAD Diagnostic Test" << std::endl;
    std::cout << "=========================" << std::endl;

    try
    {
        // Create a sphere source (uses vtkFiltersCore)
        vtkSmartPointer<vtkSphereSource> sphereSource =
            vtkSmartPointer<vtkSphereSource>::New();
        sphereSource->SetRadius(1.0);
        sphereSource->SetThetaResolution(30);
        sphereSource->SetPhiResolution(30);
        sphereSource->Update();

        std::cout << "Sphere source created successfully." << std::endl;

        // Create a mapper (uses vtkRenderingCore)
        vtkSmartPointer<vtkPolyDataMapper> mapper =
            vtkSmartPointer<vtkPolyDataMapper>::New();
        mapper->SetInputConnection(sphereSource->GetOutputPort());

        std::cout << "Mapper created successfully." << std::endl;

        // Create an actor
        vtkSmartPointer<vtkActor> actor =
            vtkSmartPointer<vtkActor>::New();
        actor->SetMapper(mapper);
        actor->GetProperty()->SetColor(0.0, 0.8, 1.0); // Light blue

        std::cout << "Actor created successfully." << std::endl;

        // Create a renderer (uses vtkRenderingCore)
        vtkSmartPointer<vtkRenderer> renderer =
            vtkSmartPointer<vtkRenderer>::New();
        renderer->AddActor(actor);
        renderer->SetBackground(0.1, 0.1, 0.1); // Dark background

        std::cout << "Renderer created successfully." << std::endl;

        // Create a render window (uses vtkRenderingOpenGL2 and vtkglad)
        // This is where GLAD symbols are typically needed
        vtkSmartPointer<vtkRenderWindow> renderWindow =
            vtkSmartPointer<vtkRenderWindow>::New();
        renderWindow->AddRenderer(renderer);
        renderWindow->SetSize(800, 600);
        renderWindow->SetWindowName("VTK GLAD Test");

        std::cout << "Render window created successfully." << std::endl;

        // Initialize the OpenGL context - THIS IS THE CRITICAL PART
        // This will trigger GLAD initialization and OpenGL function loading
        std::cout << "Initializing OpenGL context..." << std::endl;
        renderWindow->Render();

        std::cout << "\n=== SUCCESS ===" << std::endl;
        std::cout << "OpenGL context initialized successfully!" << std::endl;
        std::cout << "VTK Rendering system: " << renderWindow->GetRenderingBackend() << std::endl;

        // Get OpenGL version info
        vtkOpenGLRenderWindow* glRenderWindow =
            vtkOpenGLRenderWindow::SafeDownCast(renderWindow);
        if (glRenderWindow)
        {
            std::cout << "OpenGL Vendor: " << glRenderWindow->GetRenderingBackend() << std::endl;
        }

        std::cout << "\nPress Enter to open render window (will show for 3 seconds)..." << std::endl;
        std::cin.get();

        // Create an interactor
        glDisable(GL_DEPTH_TEST);
        vtkSmartPointer<vtkRenderWindowInteractor> interactor =
            vtkSmartPointer<vtkRenderWindowInteractor>::New();
        interactor->SetRenderWindow(renderWindow);
        glEnable(GL_DEPTH_TEST);

        // Render and show briefly
        renderWindow->Render();

        std::cout << "Window displayed. Closing in 3 seconds..." << std::endl;

        // Simple delay without interaction
        for (int i = 0; i < 3; ++i)
        {
            renderWindow->Render();
            std::this_thread::sleep_for(std::chrono::seconds(10));
        }

        std::cout << "\nTest completed successfully!" << std::endl;
    }
    catch (const std::exception& e)
    {
        std::cerr << "\n=== ERROR ===" << std::endl;
        std::cerr << "Exception caught: " << e.what() << std::endl;
        return 1;
    }
    catch (...)
    {
        std::cerr << "\n=== ERROR ===" << std::endl;
        std::cerr << "Unknown exception caught!" << std::endl;
        return 1;
    }

    return 0;
}
